/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* CURSOR.C --- Combinable craphics cursors. */
#include <stdlib.h>
#include <assert.h>
#include <mem.h>
#include <graphics.h>
#include <stdarg.h>
#include "window.h"

#pragma warn -par

static CURSOR in_cursor, protected_cursor;

LOCAL(void) draw_hairs(CURSOR c)
{
  line(c->x, 0, c->x, c->window.height - 1);
  line(0, c->y, c->window.width - 1, c->y);
}

LOCAL(void) draw_rect(CURSOR c)
{
  int x0 = c->x - c->x0, y0 = c->y - c->y0;
  int x1 = x0 + c->width - 1, y1 = y0 + c->height - 1;
  int t;

  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }
  rectangle(x0, y0, x1, y1);
}

LOCAL(void) draw_box(CURSOR c)
{
  int x0 = c->x, y0 = c->y;
  int x1 = c->x0, y1 = c->y0;
  int t;

  if (x0 > x1) { t = x0; x0 = x1; x1 = t; }
  if (y0 > y1) { t = y0; y0 = y1; y1 = t; }
  rectangle(x0, y0, x1, y1);
}

LOCAL(void) draw_line(CURSOR c)
{
  line(c->x0, c->y0, c->x, c->y);
}

LOCAL(void) draw_vector(CURSOR c)
{
  line(c->x, c->y, c->x0, c->y0);
  if (c_status_p(c, cVARIANT)) 
    arrowhead(c->x, c->y, c->x0, c->y0);
  else
    arrowhead(c->x0, c->y0, c->x, c->y);
}

LOCAL(void) draw_circle_rad(CURSOR c)
{
  arcs(c->x0, c->y0, isqrt(sum_sqr(c->x - c->x0, c->y - c->y0)), 0xf);
}

LOCAL(void) draw_circle_diam(CURSOR c)
{
  int xc = (c->x0 + c->x) >> 1,
      yc = (c->y0 + c->y) >> 1;
  arcs(xc, yc, isqrt(sum_sqr(c->x - c->x0, c->y - c->y0)) >> 1, 0xf);
}

LOCAL(void) draw_oval(CURSOR c)
{
  oval(c->x0, c->y0, c->x, c->y, c->oval_rad, c->oval_mask = 0xf);
}

/* Auxiliary for the two that follow. */
LOCAL(void) draw_oval4(CURSOR c, char near tbl[][2][2])
{
  oval(c->x0, c->y0, c->x, c->y, c->oval_rad, 
       c->oval_mask = tbl[c_status_p(c, cVARIANT) != 0][c->x > c->x0][c->y > c->y0]);
}

LOCAL(void) draw_oval2h(CURSOR c)
{
  static char otbl[2][2][2] = { 6, 6, 9, 9, 9, 9, 6, 6 };
  draw_oval4(c, otbl);
}

LOCAL(void) draw_oval2v(CURSOR c)
{
  static char otbl[2][2][2] = { 12, 3, 12, 3, 3, 12, 3, 12 };
  draw_oval4(c, otbl);
}

LOCAL(void) draw_oval4h(CURSOR c)
{
  static char otbl[2][2][2] = { 2, 4, 1, 8, 4, 2, 8, 1};
  draw_oval4(c, otbl);
}


LOCAL(void) draw_oval4v(CURSOR c)
{
  static char otbl[2][2][2] = { 8, 1, 4, 2, 1, 8, 2, 4 };
  draw_oval4(c, otbl);
}

TYPEDEF_LOCAL(void) (*DRAW_CURSOR_FUN)(CURSOR);

static DRAW_CURSOR_FUN draw_cursor_tbl[] = {
  draw_hairs,
  draw_rect,
  draw_box,
  draw_line,
  draw_vector,
  draw_circle_rad,
  draw_circle_diam,
  draw_oval, 
  draw_oval2h,
  draw_oval2v,
  draw_oval4h, 
  draw_oval4v, };

LOCAL(void) draw_cursor(CURSOR c)
{
  unsigned mask;
  DRAW_CURSOR_FUN *draw;

  if (c->level <= 0)
    return;

  push_graphics_state(&c->window, 1);
  setlinestyle(SOLID_LINE, 0, 1);
  setcolor(c->color);
  setwritemode(1);
  push_mouse_cursor_state(0);

  for (mask = c->status & ~(~0 << (cOVAL4V + 1)), draw = draw_cursor_tbl; 
       mask; 
       mask >>= 1, ++draw)
    if (mask & 1)
      (**draw)(c);

  pop_mouse_cursor_state();
  setwritemode(0);
  pop_graphics_state();
}

LOCAL(void) handle_entry(EVENT e)
{
  CURSOR c = (CURSOR)e->mouse.window;
  push_mouse_cursor_state(c_status_p(c, cMOUSE));
  c->x = e->mouse.x;
  c->y = e->mouse.y;
  c->status |= bit(cIN);
  if (c->constraint != NULL)
    (*c->constraint)(c);
  draw_cursor(c);
  in_cursor = c;
}

LOCAL(void) handle_departure(EVENT e)
{
  CURSOR c = (CURSOR)e->mouse.window;
  draw_cursor(c);
  c->status &= notbit(cIN);
  pop_mouse_cursor_state();
  in_cursor = NULL;
}

LOCAL(void) handle_motion(EVENT e)
{
  CURSOR c = (CURSOR)e->mouse.window;
  draw_cursor(c);
  c->x = e->mouse.x;
  c->y = e->mouse.y;
  if (c->constraint != NULL)
    (*c->constraint)(c);
  draw_cursor(c);
}

#define c ((CURSOR)e->mouse.window)
BeginDefDispatch(cursor)
  Dispatch(eENTRY, handle_entry)
  Dispatch(eDEPARTURE, handle_departure)
  Dispatch(eMOUSE_MOTION, handle_motion)
  DispatchAction(eBUTTON_PRESS, (*c->press)(e))
  DispatchAction(eBUTTON_RELEASE, (*c->release)(e))
EndDefDispatch(cursor)
#undef c

void open_cursor(CURSOR c, int x, int y, int width, int height, int color, WINDOW parent)
{
  memset(c, 0, sizeof(CURSOR_REC));
  open_window(&c->window, parent, x, y, width, height, 
	      0, TRANSPARENT, TRANSPARENT,
	      bit(eENTRY)|bit(eDEPARTURE)|bit(eMOUSE_MOTION)|
	      bit(eBUTTON_PRESS)|bit(eBUTTON_RELEASE));
  assert(!colorless_p(root_window->bg_color));
  while (colorless_p(parent->bg_color))
    parent = parent->parent;
  c->press = c->release = null_handler_code;
  c->constraint = NULL;
  SetDispatch(&c->window, cursor);
  set_cursor_color(c, color);
}

void show_cursor(CURSOR c, int p)
{
  if (p)
    ++c->level;
  if (c->level == 1 && c_status_p(c, cIN)) {
    if (p && c->constraint != NULL) 
      (*c->constraint)(c);
    draw_cursor(c);
  }
  if (!p)
    --c->level;
# define MOUSE_IN (bit(cMOUSE)|bit(cIN))
  if ((c->status & MOUSE_IN) == MOUSE_IN)
    show_mouse_cursor(p);
}

LOCAL(void) unprotect_graphic_cursor(void)
{
  if (protected_cursor != NULL) 
    show_cursor(protected_cursor, 1);
  protected_cursor = NULL;
}

LOCAL(void) protect_graphic_cursor(int x0, int y0, int x1, int y1)
{
  if (protected_cursor != NULL)
    unprotect_graphic_cursor();

  if (in_cursor != NULL &&
      in_cursor->level > 0 &&
      (in_cursor->status & GRAPHIC_CURSORS) &&
      x0 < in_cursor->window.x + in_cursor->window.width && 
      in_cursor->window.x < x1 &&
      y0 < in_cursor->window.y + in_cursor->window.height && 
      in_cursor->window.y < y1) {
    protected_cursor = in_cursor;
    show_cursor(in_cursor, 0);
  }
}

/* Turn off if it's potentially inside window w. */
void protect_cursor(WINDOW w)
{
  int bw = w->border_width,
      x0 = max(0, w->x - bw),
      y0 = max(0, w->y - bw),
      x1 = min(x_max, w->x + w->width + bw),
      y1 = min(y_max, w->y + w->height + bw);
  protect_mouse_cursor(x0, y0, x1, y1);
  protect_graphic_cursor(x0, y0, x1, y1);
}

void unprotect_cursor(void)
{
  unprotect_mouse_cursor();
  unprotect_graphic_cursor();
}

/* Set the kind of cursor.  If a two-point cursor
   is included in the mask, set its start location, also. 
   If ap is null, set the cursor mask only. */
void vset_cursor(CURSOR c, unsigned mask, va_list ap)
{
  show_cursor(c, 0);
  /* Set cursor type bits, but don't change `in' bit. */
  c->status = mask | (bit(cIN) & c->status);
  if (ap != NULL && (mask & (TWO_PT_CURSORS | bit(cRECT)))) {
    c->x0 = va_arg(ap, int);
    c->y0 = va_arg(ap, int);
    if (mask & bit(cRECT)) {
      c->width = va_arg(ap, int);
      c->height = va_arg(ap, int);
    }
    if (mask & OVAL_CURSORS)
      c->oval_rad = va_arg(ap, int);
  }
  show_cursor(c, 1);
}

/* Same as above, but accept variable argument list
   instead of pointer. */
void set_cursor(CURSOR c, unsigned mask, ...)
{
  va_list ap;
  va_start(ap, mask);
  vset_cursor(c, mask, ap);
  va_end(ap);
}

/* Set the function that constrains cursor configuration. */
void set_cursor_constraint(CURSOR c, CURSOR_CONSTRAINT fun, ENV env)
{
  show_cursor(c, 0);
  c->constraint = fun;
  c->constraint_env = env;
  show_cursor(c, 1);
}

/* Unset the cursor constraint of a cursor and return its 
   environment. Redraw the cursor at its unconstrained location. */
ENV unset_cursor_constraint(CURSOR c)
{
  ENV env = c->constraint_env;
  show_cursor(c, 0);
  c->constraint = NULL;
  c->constraint_env = NULL;
  c->x = pointer_x(&c->window);
  c->y = pointer_y(&c->window);
  show_cursor(c, 1);
  return env;
}

/* Set the color of the cursor. */
void set_cursor_color(CURSOR c, int color)
{
  show_cursor(c, 0);
  c->color = color ^ c->window.parent->bg_color;
  show_cursor(c, 1);
}

/* Get the current position of a cursor.  If a two-point
   cursor is in the status mask, return its start location, also. */
unsigned get_cursor(CURSOR c, int *x_rtn, int *y_rtn, ...)
{
  va_list ap;

  *x_rtn = c->x;
  *y_rtn = c->y;
  va_start(ap, y_rtn);
  if (c->status & TWO_PT_CURSORS) {
    *va_arg(ap, int*) = c->x0;
    *va_arg(ap, int*) = c->y0;
  }
  if (c->status & OVAL_CURSORS) 
    *va_arg(ap, unsigned*) = c->oval_mask;
  va_end(ap);
  return c->status & ~bit(cIN);
}
