/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.cli;

import com.google.common.base.Throwables;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.prefs.BackingStoreException;
import org.jabref.Globals;
import org.jabref.JabRefException;
import org.jabref.cli.AuxCommandLine;
import org.jabref.cli.ImportInspectionCommandLine;
import org.jabref.cli.JabRefCLI;
import org.jabref.cli.SystemOutputPrinter;
import org.jabref.gui.externalfiles.AutoSetLinks;
import org.jabref.gui.importer.fetcher.EntryFetcher;
import org.jabref.gui.importer.fetcher.EntryFetchers;
import org.jabref.logic.bibtexkeypattern.BibtexKeyGenerator;
import org.jabref.logic.exporter.BibtexDatabaseWriter;
import org.jabref.logic.exporter.Exporter;
import org.jabref.logic.exporter.ExporterFactory;
import org.jabref.logic.exporter.FileSaveSession;
import org.jabref.logic.exporter.SaveException;
import org.jabref.logic.exporter.SavePreferences;
import org.jabref.logic.exporter.SaveSession;
import org.jabref.logic.exporter.TemplateExporter;
import org.jabref.logic.importer.ImportException;
import org.jabref.logic.importer.ImportFormatReader;
import org.jabref.logic.importer.OpenDatabase;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.fileformat.BibtexParser;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.layout.LayoutFormatterPreferences;
import org.jabref.logic.logging.JabRefLogger;
import org.jabref.logic.net.URLDownload;
import org.jabref.logic.search.DatabaseSearcher;
import org.jabref.logic.search.SearchQuery;
import org.jabref.logic.shared.prefs.SharedDatabasePreferences;
import org.jabref.logic.util.OS;
import org.jabref.logic.xmp.XmpPreferences;
import org.jabref.model.Defaults;
import org.jabref.model.EntryTypes;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.strings.StringUtil;
import org.jabref.preferences.SearchPreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArgumentProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(ArgumentProcessor.class);
    private final JabRefCLI cli;
    private final List<ParserResult> parserResults;
    private final Mode startupMode;
    private boolean noGUINeeded;

    public ArgumentProcessor(String[] args, Mode startupMode) {
        this.cli = new JabRefCLI(args);
        this.startupMode = startupMode;
        this.parserResults = this.processArguments();
    }

    private static Optional<ParserResult> importToOpenBase(String argument) {
        Optional<ParserResult> result = ArgumentProcessor.importFile(argument);
        result.ifPresent(ParserResult::setToOpenTab);
        return result;
    }

    private static Optional<ParserResult> importBibtexToOpenBase(String argument) {
        BibtexParser parser = new BibtexParser(Globals.prefs.getImportFormatPreferences(), Globals.getFileUpdateMonitor());
        try {
            List<BibEntry> entries = parser.parseEntries(argument);
            ParserResult result = new ParserResult(entries);
            result.setToOpenTab();
            return Optional.of(result);
        }
        catch (ParseException e) {
            System.err.println(Localization.lang("Error occurred when parsing entry", new String[0]) + ": " + e.getLocalizedMessage());
            return Optional.empty();
        }
    }

    private static Optional<ParserResult> importFile(String argument) {
        Path file;
        String[] data = argument.split(",");
        String address = data[0];
        if (address.startsWith("http://") || address.startsWith("https://") || address.startsWith("ftp://")) {
            try {
                file = new URLDownload(address).toTemporaryFile();
            }
            catch (IOException e) {
                System.err.println(Localization.lang("Problem downloading from %1", address) + e.getLocalizedMessage());
                return Optional.empty();
            }
        } else {
            file = OS.WINDOWS ? Paths.get(address, new String[0]) : Paths.get(address.replace("~", System.getProperty("user.home")), new String[0]);
        }
        String importFormat = data.length > 1 ? data[1] : "*";
        Optional<ParserResult> importResult = ArgumentProcessor.importFile(file, importFormat);
        importResult.ifPresent(result -> {
            SystemOutputPrinter printer = new SystemOutputPrinter();
            if (result.hasWarnings()) {
                printer.showMessage(result.getErrorMessage());
            }
        });
        return importResult;
    }

    private static Optional<ParserResult> importFile(Path file, String importFormat) {
        try {
            if (!"*".equals(importFormat)) {
                System.out.println(Localization.lang("Importing", new String[0]) + ": " + file);
                ParserResult result = Globals.IMPORT_FORMAT_READER.importFromFile(importFormat, file);
                return Optional.of(result);
            }
            System.out.println(Localization.lang("Importing in unknown format", new String[0]) + ": " + file);
            ImportFormatReader.UnknownFormatImport importResult = Globals.IMPORT_FORMAT_READER.importUnknownFormat(file, Globals.getFileUpdateMonitor());
            System.out.println(Localization.lang("Format used", new String[0]) + ": " + importResult.format);
            return Optional.of(importResult.parserResult);
        }
        catch (ImportException ex) {
            System.err.println(Localization.lang("Error opening file", new String[0]) + " '" + file + "': " + ex.getLocalizedMessage());
            return Optional.empty();
        }
    }

    public List<ParserResult> getParserResults() {
        return this.parserResults;
    }

    public boolean hasParserResults() {
        return !this.parserResults.isEmpty();
    }

    private List<ParserResult> processArguments() {
        if (!this.cli.isBlank() && this.cli.isDebugLogging()) {
            JabRefLogger.setDebug();
        }
        if (this.startupMode == Mode.INITIAL_START && this.cli.isShowVersion()) {
            this.cli.displayVersion();
        }
        if (this.startupMode == Mode.INITIAL_START && this.cli.isHelp()) {
            this.cli.printUsage();
            this.noGUINeeded = true;
            return Collections.emptyList();
        }
        if (this.cli.isPreferencesReset()) {
            this.resetPreferences(this.cli.getPreferencesReset());
        }
        if (this.cli.isPreferencesImport()) {
            this.importPreferences();
        }
        List<ParserResult> loaded = this.importAndOpenFiles();
        if (!this.cli.isBlank() && this.cli.isFetcherEngine()) {
            this.fetch(this.cli.getFetcherEngine()).ifPresent(loaded::add);
        }
        if (this.cli.isExportMatches()) {
            if (!loaded.isEmpty()) {
                if (!this.exportMatches(loaded)) {
                    return Collections.emptyList();
                }
            } else {
                System.err.println(Localization.lang("The output option depends on a valid input option.", new String[0]));
            }
        }
        if (this.cli.isGenerateBibtexKeys()) {
            this.regenerateBibtexKeys(loaded);
        }
        if (this.cli.isAutomaticallySetFileLinks()) {
            this.automaticallySetFileLinks(loaded);
        }
        if (this.cli.isFileExport()) {
            if (!loaded.isEmpty()) {
                this.exportFile(loaded, this.cli.getFileExport().split(","));
            } else {
                System.err.println(Localization.lang("The output option depends on a valid import option.", new String[0]));
            }
        }
        LOGGER.debug("Finished export");
        if (this.cli.isPreferencesExport()) {
            try {
                Globals.prefs.exportPreferences(this.cli.getPreferencesExport());
            }
            catch (JabRefException ex) {
                LOGGER.error("Cannot export preferences", ex);
            }
        }
        if (!this.cli.isBlank() && this.cli.isAuxImport()) {
            this.doAuxImport(loaded);
        }
        return loaded;
    }

    private boolean exportMatches(List<ParserResult> loaded) {
        String[] data = this.cli.getExportMatches().split(",");
        String searchTerm = data[0].replace("\\$", " ");
        ParserResult pr = loaded.get(loaded.size() - 1);
        BibDatabaseContext databaseContext = pr.getDatabaseContext();
        BibDatabase dataBase = pr.getDatabase();
        SearchPreferences searchPreferences = new SearchPreferences(Globals.prefs);
        SearchQuery query = new SearchQuery(searchTerm, searchPreferences.isCaseSensitive(), searchPreferences.isRegularExpression());
        List<BibEntry> matches = new DatabaseSearcher(query, dataBase).getMatches();
        if (!matches.isEmpty()) {
            String formatName;
            switch (data.length) {
                case 3: {
                    formatName = data[2];
                    break;
                }
                case 2: {
                    formatName = "tablerefsabsbib";
                    break;
                }
                default: {
                    System.err.println(Localization.lang("Output file missing", new String[0]).concat(". \n \t ").concat(Localization.lang("Usage", new String[0])).concat(": ") + JabRefCLI.getExportMatchesSyntax());
                    this.noGUINeeded = true;
                    return false;
                }
            }
            Optional<Exporter> exporter = Globals.exportFactory.getExporterByName(formatName);
            if (!exporter.isPresent()) {
                System.err.println(Localization.lang("Unknown export format", new String[0]) + ": " + formatName);
            } else {
                try {
                    System.out.println(Localization.lang("Exporting", new String[0]) + ": " + data[1]);
                    exporter.get().export(databaseContext, Paths.get(data[1], new String[0]), databaseContext.getMetaData().getEncoding().orElse(Globals.prefs.getDefaultEncoding()), matches);
                }
                catch (Exception ex) {
                    System.err.println(Localization.lang("Could not export file", new String[0]) + " '" + data[1] + "': " + Throwables.getStackTraceAsString(ex));
                }
            }
        } else {
            System.err.println(Localization.lang("No search matches.", new String[0]));
        }
        return true;
    }

    private void doAuxImport(List<ParserResult> loaded) {
        boolean usageMsg = !loaded.isEmpty() ? this.generateAux(loaded, this.cli.getAuxImport().split(",")) : true;
        if (usageMsg) {
            System.out.println(Localization.lang("no base-BibTeX-file specified", new String[0]) + "!");
            System.out.println(Localization.lang("usage", new String[0]) + " :");
            System.out.println("jabref --aux infile[.aux],outfile[.bib] base-BibTeX-file");
        }
    }

    private List<ParserResult> importAndOpenFiles() {
        ArrayList<ParserResult> loaded = new ArrayList<ParserResult>();
        ArrayList<String> toImport = new ArrayList<String>();
        if (!this.cli.isBlank() && !this.cli.getLeftOver().isEmpty()) {
            for (String aLeftOver : this.cli.getLeftOver()) {
                boolean bibExtension = aLeftOver.toLowerCase(Locale.ENGLISH).endsWith("bib");
                ParserResult pr = new ParserResult();
                if (bibExtension) {
                    pr = OpenDatabase.loadDatabase(aLeftOver, Globals.prefs.getImportFormatPreferences(), Globals.getFileUpdateMonitor());
                }
                if (!bibExtension || pr.isEmpty()) {
                    if (this.startupMode == Mode.INITIAL_START) {
                        toImport.add(aLeftOver);
                        continue;
                    }
                    loaded.add(ArgumentProcessor.importToOpenBase(aLeftOver).orElse(new ParserResult()));
                    continue;
                }
                loaded.add(pr);
            }
        }
        if (!this.cli.isBlank() && this.cli.isFileImport()) {
            toImport.add(this.cli.getFileImport());
        }
        for (String filenameString : toImport) {
            ArgumentProcessor.importFile(filenameString).ifPresent(loaded::add);
        }
        if (!this.cli.isBlank() && this.cli.isImportToOpenBase()) {
            ArgumentProcessor.importToOpenBase(this.cli.getImportToOpenBase()).ifPresent(loaded::add);
        }
        if (!this.cli.isBlank() && this.cli.isBibtexImport()) {
            ArgumentProcessor.importBibtexToOpenBase(this.cli.getBibtexImport()).ifPresent(loaded::add);
        }
        return loaded;
    }

    private boolean generateAux(List<ParserResult> loaded, String[] data) {
        if (data.length == 2) {
            ParserResult pr = loaded.get(0);
            AuxCommandLine acl = new AuxCommandLine(data[0], pr.getDatabase());
            BibDatabase newBase = acl.perform();
            boolean notSavedMsg = false;
            if (newBase != null && newBase.hasEntries()) {
                String subName = StringUtil.getCorrectFileName(data[1], "bib");
                try {
                    System.out.println(Localization.lang("Saving", new String[0]) + ": " + subName);
                    SavePreferences prefs = Globals.prefs.loadForSaveFromPreferences();
                    BibtexDatabaseWriter<SaveSession> databaseWriter = new BibtexDatabaseWriter<SaveSession>(FileSaveSession::new);
                    Defaults defaults = new Defaults(Globals.prefs.getDefaultBibDatabaseMode());
                    Object session = databaseWriter.saveDatabase(new BibDatabaseContext(newBase, defaults), prefs);
                    if (!((SaveSession)session).getWriter().couldEncodeAll()) {
                        System.err.println(Localization.lang("Warning", new String[0]) + ": " + Localization.lang("The chosen encoding '%0' could not encode the following characters:", ((SaveSession)session).getEncoding().displayName()) + " " + ((SaveSession)session).getWriter().getProblemCharacters());
                    }
                    ((SaveSession)session).commit(subName);
                }
                catch (SaveException ex) {
                    System.err.println(Localization.lang("Could not save file.", new String[0]) + "\n" + ex.getLocalizedMessage());
                }
                notSavedMsg = true;
            }
            if (!notSavedMsg) {
                System.out.println(Localization.lang("no library generated", new String[0]));
            }
            return false;
        }
        return true;
    }

    private void exportFile(List<ParserResult> loaded, String[] data) {
        if (data.length == 1) {
            if (!loaded.isEmpty()) {
                ParserResult pr = loaded.get(loaded.size() - 1);
                if (!pr.isInvalid()) {
                    try {
                        System.out.println(Localization.lang("Saving", new String[0]) + ": " + data[0]);
                        SavePreferences prefs = Globals.prefs.loadForSaveFromPreferences();
                        Defaults defaults = new Defaults(Globals.prefs.getDefaultBibDatabaseMode());
                        BibtexDatabaseWriter<SaveSession> databaseWriter = new BibtexDatabaseWriter<SaveSession>(FileSaveSession::new);
                        Object session = databaseWriter.saveDatabase(new BibDatabaseContext(pr.getDatabase(), pr.getMetaData(), defaults), prefs);
                        if (!((SaveSession)session).getWriter().couldEncodeAll()) {
                            System.err.println(Localization.lang("Warning", new String[0]) + ": " + Localization.lang("The chosen encoding '%0' could not encode the following characters:", ((SaveSession)session).getEncoding().displayName()) + " " + ((SaveSession)session).getWriter().getProblemCharacters());
                        }
                        ((SaveSession)session).commit(data[0]);
                    }
                    catch (SaveException ex) {
                        System.err.println(Localization.lang("Could not save file.", new String[0]) + "\n" + ex.getLocalizedMessage());
                    }
                }
            } else {
                System.err.println(Localization.lang("The output option depends on a valid import option.", new String[0]));
            }
        } else if (data.length == 2) {
            ParserResult pr = loaded.get(loaded.size() - 1);
            File theFile = pr.getFile().get();
            if (!theFile.isAbsolute()) {
                theFile = theFile.getAbsoluteFile();
            }
            BibDatabaseContext databaseContext = pr.getDatabaseContext();
            databaseContext.setDatabaseFile(theFile);
            Globals.prefs.fileDirForDatabase = databaseContext.getFileDirectories(Globals.prefs.getFileDirectoryPreferences());
            System.out.println(Localization.lang("Exporting", new String[0]) + ": " + data[0]);
            Optional<Exporter> exporter = Globals.exportFactory.getExporterByName(data[1]);
            if (!exporter.isPresent()) {
                System.err.println(Localization.lang("Unknown export format", new String[0]) + ": " + data[1]);
            } else {
                try {
                    exporter.get().export(pr.getDatabaseContext(), Paths.get(data[0], new String[0]), pr.getDatabaseContext().getMetaData().getEncoding().orElse(Globals.prefs.getDefaultEncoding()), (List<BibEntry>)pr.getDatabaseContext().getDatabase().getEntries());
                }
                catch (Exception ex) {
                    System.err.println(Localization.lang("Could not export file", new String[0]) + " '" + data[0] + "': " + Throwables.getStackTraceAsString(ex));
                }
            }
        }
    }

    private void importPreferences() {
        try {
            Globals.prefs.importPreferences(this.cli.getPreferencesImport());
            EntryTypes.loadCustomEntryTypes(Globals.prefs.loadCustomEntryTypes(BibDatabaseMode.BIBTEX), Globals.prefs.loadCustomEntryTypes(BibDatabaseMode.BIBLATEX));
            Map<String, TemplateExporter> customExporters = Globals.prefs.customExports.getCustomExportFormats(Globals.prefs, Globals.journalAbbreviationLoader);
            LayoutFormatterPreferences layoutPreferences = Globals.prefs.getLayoutFormatterPreferences(Globals.journalAbbreviationLoader);
            SavePreferences savePreferences = Globals.prefs.loadForExportFromPreferences();
            XmpPreferences xmpPreferences = Globals.prefs.getXMPPreferences();
            Globals.exportFactory = ExporterFactory.create(customExporters, layoutPreferences, savePreferences, xmpPreferences);
        }
        catch (JabRefException ex) {
            LOGGER.error("Cannot import preferences", ex);
        }
    }

    private void resetPreferences(String value) {
        if ("all".equals(value.trim())) {
            try {
                System.out.println(Localization.lang("Setting all preferences to default values.", new String[0]));
                Globals.prefs.clear();
                new SharedDatabasePreferences().clear();
            }
            catch (BackingStoreException e) {
                System.err.println(Localization.lang("Unable to clear preferences.", new String[0]));
                LOGGER.error("Unable to clear preferences", e);
            }
        } else {
            String[] keys;
            for (String key : keys = value.split(",")) {
                if (Globals.prefs.hasKey(key.trim())) {
                    System.out.println(Localization.lang("Resetting preference key '%0'", key.trim()));
                    Globals.prefs.clear(key.trim());
                    continue;
                }
                System.out.println(Localization.lang("Unknown preference key '%0'", key.trim()));
            }
        }
    }

    private void automaticallySetFileLinks(List<ParserResult> loaded) {
        for (ParserResult parserResult : loaded) {
            BibDatabase database = parserResult.getDatabase();
            LOGGER.info(Localization.lang("Automatically setting file links", new String[0]));
            AutoSetLinks.autoSetLinks(database.getEntries(), parserResult.getDatabaseContext());
        }
    }

    private void regenerateBibtexKeys(List<ParserResult> loaded) {
        for (ParserResult parserResult : loaded) {
            BibDatabase database = parserResult.getDatabase();
            LOGGER.info(Localization.lang("Regenerating BibTeX keys according to metadata", new String[0]));
            BibtexKeyGenerator keyGenerator = new BibtexKeyGenerator(parserResult.getDatabaseContext(), Globals.prefs.getBibtexKeyPatternPreferences());
            for (BibEntry entry : database.getEntries()) {
                keyGenerator.generateAndSetKey(entry);
            }
        }
    }

    private Optional<ParserResult> fetch(String fetchCommand) {
        if (fetchCommand == null || !fetchCommand.contains(":") || fetchCommand.split(":").length != 2) {
            System.out.println(Localization.lang("Expected syntax for --fetch='<name of fetcher>:<query>'", new String[0]));
            System.out.println(Localization.lang("The following fetchers are available:", new String[0]));
            return Optional.empty();
        }
        String[] split = fetchCommand.split(":");
        String engine = split[0];
        EntryFetchers fetchers = new EntryFetchers(Globals.journalAbbreviationLoader);
        EntryFetcher fetcher = null;
        for (EntryFetcher e : fetchers.getEntryFetchers()) {
            if (!engine.equalsIgnoreCase(e.getClass().getSimpleName().replace("Fetcher", ""))) continue;
            fetcher = e;
        }
        if (fetcher == null) {
            System.out.println(Localization.lang("Could not find fetcher '%0'", engine));
            System.out.println(Localization.lang("The following fetchers are available:", new String[0]));
            for (EntryFetcher e : fetchers.getEntryFetchers()) {
                System.out.println("  " + e.getClass().getSimpleName().replace("Fetcher", "").toLowerCase(Locale.ENGLISH));
            }
            return Optional.empty();
        }
        String query = split[1];
        System.out.println(Localization.lang("Running query '%0' with fetcher '%1'.", query, engine) + " " + Localization.lang("Please wait...", new String[0]));
        Collection<BibEntry> result = new ImportInspectionCommandLine().query(query, fetcher);
        if (result.isEmpty()) {
            System.out.println(Localization.lang("Query '%0' with fetcher '%1' did not return any results.", query, engine));
            return Optional.empty();
        }
        return Optional.of(new ParserResult(result));
    }

    public boolean isBlank() {
        return this.cli.isBlank();
    }

    public boolean shouldShutDown() {
        return this.cli.isDisableGui() || this.cli.isShowVersion() || this.noGUINeeded;
    }

    public static enum Mode {
        INITIAL_START,
        REMOTE_START;

    }
}

