/*
 * Copyright (C) 2002-2008 The Warp Rogue Team
 * Part of the Warp Rogue Project
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY.
 *
 * See the license.txt file for more details.
 */

/*
 * Module: AI Equipment
 */

#include "wrogue.h"


static bool     ai_weapon_suitable(CHARACTER *, const OBJECT *);

static AI_OPTION_WEIGHT ai_weapon_weight_base(CHARACTER *, const OBJECT *);

static void     ai_weapon_weight_modifier(CHARACTER *, const OBJECT *,
                        AI_OPTION_WEIGHT *
                );

static void	ai_unarmed_weight_modifier(CHARACTER *, const OBJECT *, 
                        AI_OPTION_WEIGHT *
                );

static void	ai_ccw_weight_modifier(CHARACTER *, const OBJECT *, 
                        AI_OPTION_WEIGHT *
                );

static void	ai_rcw_weight_modifier(CHARACTER *,
                        const OBJECT *, AI_OPTION_WEIGHT *
                );



/* Chooses a firing mode
 */
void ai_choose_firing_mode(CHARACTER * self,
	FIRING_DATA * firing_data
)
{
	const FIRING_MODE_DATA *firing_mode =
		&self->weapon->firing_mode;

	if (firing_mode->has[FMODE_A]) {

		firing_data->firing_mode = FMODE_A;

		firing_data->n_shots = firing_mode->a_shots;

		firing_data->spread = rng_r(
			A_MODE_SPREAD_MIN, A_MODE_SPREAD_MAX
		);

	} else if (firing_mode->has[FMODE_SA]) {

		firing_data->firing_mode = FMODE_SA;

		firing_data->n_shots = rng_r(
			firing_mode->min_sa_shots,
			firing_mode->max_sa_shots
		);

	} else if (firing_mode->has[FMODE_S]) {

		firing_data->firing_mode = FMODE_S;
		firing_data->n_shots = 1;
	}
}



/* Returns the AI option weight of a weapon
 */
AI_OPTION_WEIGHT ai_weapon_weight(CHARACTER *self, const OBJECT *weapon)
{
        AI_OPTION_WEIGHT weight;

        if (!ai_weapon_suitable(self, weapon)) return AI_OPTION_WEIGHT_NIL;

        weight = ai_weapon_weight_base(self, weapon);

        ai_weapon_weight_modifier(self, weapon, &weight);

        return weight;
}



/* Returns true if a weapon is suitable
 */
static bool ai_weapon_suitable(CHARACTER *self, const OBJECT *weapon)
{
        if (weapon == NULL || weapon->type == OTYPE_CLOSE_COMBAT_WEAPON) {
                
                if (!self->target.reachable) return false;

                if (character_target_distance(self) > 1 &&
                        (self->tactic == TACTIC_STAY_CLOSE ||
                                self->tactic == TACTIC_HOLD_POSITION ||
                        movement_speed_max(self) == 0)) return false;

                if (weapon == NULL && character_has_attribute(self, 
                        CA_NO_UNARMED_ATTACK)) return false;

        } else if (weapon->type == OTYPE_RANGED_COMBAT_WEAPON) {

                if (character_target_distance(self) == 1 &&
                        weapon->subtype != OSTYPE_PISTOL) {

                        return false;
                }
        } 

        return true;
}



/* Returns the basic AI option weight of a weapon
 */
static AI_OPTION_WEIGHT ai_weapon_weight_base(CHARACTER *self,
        const OBJECT *weapon
)
{
        AI_OPTION_WEIGHT weight = 0;
        
        /* No weapon - return average unarmed damage */
        if (weapon == NULL) {
                DICE_ROLL damage_dice;

                weight += dice_roll_average(
                        unarmed_damage_power(self, &damage_dice)
                );

                return weight;
        }

        /* Weight is based on average damage */
        weight += dice_roll_average(&weapon->damage);

        /* Increase weight of force weapons */
        if (object_has_attribute(weapon, OA_FORCE)) {
                
                weight += 5;
        
                if (object_has_attribute(weapon, OA_FORCE_RUNE)) 
                        weight += 3;
        };

        return weight;
}



/* Applies the weapon type specific AI option weight modifier
 */
static void ai_weapon_weight_modifier(CHARACTER *self,
        const OBJECT *weapon, AI_OPTION_WEIGHT *weight
)
{
        if (weapon == NULL) {
                        
                ai_unarmed_weight_modifier(self, weapon, weight);

        } else if (weapon->type == OTYPE_CLOSE_COMBAT_WEAPON) {
                
                ai_ccw_weight_modifier(self, weapon, weight);

        } else if (weapon->type == OTYPE_RANGED_COMBAT_WEAPON) {
                
                ai_rcw_weight_modifier(self, weapon, weight);
        }
}



/* Applies the unarmed combat specific AI option weight modifier
 */
static void ai_unarmed_weight_modifier(CHARACTER *self, const OBJECT *weapon,
        AI_OPTION_WEIGHT *weight
)
{
        NOT_USED(weapon);
        
        /* More than two turns to reach the target .. */
        if (character_target_distance(self) >  
                movement_speed_max(self) * 2) {

                /* .. halve weight if possible */
                if (*weight >= 2) *weight /= 2;
        }
}



/* Applies the CCW specific AI option weight modifier
 */
static void ai_ccw_weight_modifier(CHARACTER *self, const OBJECT *weapon,
        AI_OPTION_WEIGHT *weight
)
{
        NOT_USED(weapon);
        
        /* More than two turns to reach the target .. */
        if (character_target_distance(self) >  
                movement_speed_max(self) * 2) {

                /* .. halve weight if possible */
                if (*weight >= 2) *weight /= 2;
        }
}



/* Applies the RCW specific AI option weight modifier
 */
static void ai_rcw_weight_modifier(CHARACTER *self,
        const OBJECT *weapon, AI_OPTION_WEIGHT *weight
)
{
        /* Rapid fire weapons prefered, unless the character has
         * the "Critical shot" perk
         */
        if (!character_has_perk(self, PK_CRITICAL_SHOT)) {
                const bool *has = weapon->firing_mode.has;

                if (has[FMODE_A]) {
                        
                        *weight += 10;

                } else if (has[FMODE_SA]) {

                        *weight += 5;
                }
        }

        /* Loaded weapons prefered */
        if (weapon->charge > 0) *weight += 1;

        /* Unjammed weapons prefered */
        if (!object_has_attribute(weapon, OA_JAMMED)) *weight += 1;

        /* Basic and heavy weapons prefered for long range shots */
        if (character_target_distance(self) > MAX_MEDIUM_RANGE) {

                if (weapon->subtype == OSTYPE_BASIC ||
                        weapon->subtype == OSTYPE_HEAVY) {

                        *weight += 5;
                }
        }
}

